package com.mlc.nordic_sdk.viewModel

import android.util.Log
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.mlc.nordic_sdk.bluetooth.BluetoothManager
import com.mlc.nordic_sdk.bluetooth.data.DeviceType
import com.mlc.nordic_sdk.Method
import com.mlc.nordic_sdk.protocol.protocol_code.WBPProtocol
import com.mlc.nordic_sdk.protocol.protocol_code.data.wbp.MeasurementSetting
import com.mlc.nordic_sdk.protocol.protocol_code.data.wbp.NocturnalInfo
import kotlinx.coroutines.launch

class WBPActivityViewModel: ViewModel() {
    private val TAG = "WBPActivityViewModel"
    private var bluetoothManager: BluetoothManager? = null
    private var deviceName: String? = ""
    private var deviceType: DeviceType? = null
    var state by mutableStateOf(false)

    private var wbpProtocol: WBPProtocol? = null

    private val _listData = MutableLiveData<List<String>>(emptyList())
    val listData: LiveData<List<String>>
        get() = _listData

    fun setBleReceiveManager(bluetoothManager: BluetoothManager?) {
        this.bluetoothManager = bluetoothManager

        startScan()
    }

    fun setDeviceInfo(deviceName: String, deviceType: DeviceType?) {
        this.deviceName = deviceName
        this.deviceType = deviceType
    }

    fun setWBPProtocol(listener: WBPProtocol.OnDataResponseListener) {
        // Input protocol key
        this.wbpProtocol = WBPProtocol.getInstance("", listener)
    }

    private fun startScan() {
        bluetoothManager?.startScan(DeviceType.MLC_WBP)
    }

    fun stopScan() {
        bluetoothManager?.stopScan()
    }

    //update UI
    fun addDataList(msg: String) {
        Log.e("addList", msg)

        _listData.value = _listData.value?.let { it + listOf(msg) }
    }

    fun writeCommand(data: ByteArray?) {
        addDataList("WRITE -> ${data?.let { Method.byteArrayToString(it) }}")

        viewModelScope.launch {
            bluetoothManager?.writeCommand(data)
        }
    }

    //WBP
    fun readUsualModeHistory() {
        writeCommand(wbpProtocol?.readUsualModeHistory())
    }

    fun readUsualModeHistoryEachMeasurement() {
        writeCommand(wbpProtocol?.readUsualModeHistoryEachMeasurement())
    }

    fun readDiagnosticModeHistory() {
        writeCommand(wbpProtocol?.readDiagnosticModeHistory())
    }

    fun clearSelectedModeHistory() {
        writeCommand(wbpProtocol?.clearSelectedModeHistory(usual = true, diagnostic = true))
    }

    fun clearCurrentModeHistory() {
        writeCommand(wbpProtocol?.clearCurrentModeHistory())
    }

    fun disconnectBluetooth() {
        writeCommand(wbpProtocol?.disconnectBluetooth())
    }

    fun writeDeviceTime() {
        writeCommand(wbpProtocol?.writeDeviceTime())
    }

    fun writeNewUserID() {
        writeCommand(wbpProtocol?.writeNewUserID())
    }

    fun readNocturnalModeSetting() {
        writeCommand(wbpProtocol?.readNocturnalModeSetting())
    }

    fun changeNocturnalModeSetting() {
        writeCommand(wbpProtocol?.changeNocturnalModeSetting(
            NocturnalInfo(
                openNocturnal = 1,
                startYear = 2018,
                startMonth = 11,
                startDay = 5,
                startHour = 6
            ))
        )
    }

    fun readMeasurementSetting() {
        writeCommand(wbpProtocol?.readMeasurementSetting())
    }

    fun writeMeasurementSetting() {
        writeCommand(wbpProtocol?.writeMeasurementSetting(
            MeasurementSetting(
                measurement_times = 3,
                rest_time = 0,
                interval_time_seconds = 15,
                sw_afib = 1
            )
        ))
    }

    fun readDeviceIDAndInfo() {
        writeCommand(wbpProtocol?.readDeviceIDAndInfo())
    }

    fun readDeviceTime() {
        writeCommand(wbpProtocol?.readDeviceTime())
    }

    fun readUserIDAndVersionData() {
        writeCommand(wbpProtocol?.readUserIDAndVersionData())
    }

    fun readNocturnalModeHistory() {
        writeCommand(wbpProtocol?.readNocturnalModeHistory())
    }

    fun readSerialNumber() {
        writeCommand(wbpProtocol?.readSerialNumber())
    }

    fun solveData(data: List<Byte>) {
        addDataList("NOTIFY -> ${Method.listByteToString(data)}")

        wbpProtocol?.solveDataResult(data)
    }
}