//
//  BLEDeviceInfo.h
//  MicroLifeDeviceSDK
//  藍牙設備資訊類別 / Bluetooth device information class
//  Created by willy.wu on 2020/11/19.
//

#import <Foundation/Foundation.h>
#import <MicroLifeDeviceSDK/BLESDK.h>

NS_ASSUME_NONNULL_BEGIN

@interface BLEDeviceInfo : NSObject

/// Create custom bluetooth manager 創建自定義藍牙管理器
///
/// This method allows developers to replace the default BLESDK with their own bluetooth manager implementation. 此方法允許開發者用自己的藍牙管理器實現來替換默認的 BLESDK
///
/// The custom bluetooth manager must implement BluetoothManagerProtocol to ensure compatibility. 自定義藍牙管理器必須實現 BluetoothManagerProtocol 以確保兼容性

/// @note The custom manager must implement all required methods in BluetoothManagerProtocol
///       自定義管理器必須實現 BluetoothManagerProtocol 中的所有必要方法
///
/// @warning Replacing the bluetooth manager while device is connected may cause unexpected behavior
///          在設備連接時替換藍牙管理器可能會導致意外行為
@property (nonatomic, strong) id<BluetoothManagerProtocol> sdk;

/// 設備名稱 / Device name
@property (nonatomic, strong) NSString *name;

/// 設備UUID / Device UUID
@property (nonatomic, strong) NSString *UUID;

/// MAC地址 / MAC Address
@property (nonatomic, strong) NSData *mac;

/// 信號強度 / Signal strength
@property (nonatomic, strong) NSNumber *RSSI;

/// 藍牙外圍設備 / Bluetooth peripheral device
@property (nonatomic, strong) CBPeripheral *currPeripheral;

/// 通知特徵值 / Notification characteristic
@property (nonatomic, strong) CBCharacteristic *characteristicRx;

/// GATT特徵值陣列 / GATT characteristics array
@property (nonatomic, strong) NSMutableArray *characteristicGx;

/// 寫入特徵值 / Write characteristic
@property (nonatomic, strong) CBCharacteristic *characteristicTx;

/// 廣播資料 / Advertisement data
@property (nonatomic, strong) NSDictionary *advertisementData;

/// 初始化方法 / Initialization method
/// @param baby 藍牙管理器（實現BluetoothManagerProtocol） /
///            Bluetooth manager (implements BluetoothManagerProtocol)
///            用於處理設備的掃描、連接和通訊 /
///            Handles device scanning, connection and communication
/// @param peripheral 藍牙外圍設備 / Bluetooth peripheral
/// @param advertisementData 廣播資料 / Advertisement data
/// @param RSSI 信號強度 / Signal strength
+ (instancetype)initWithDiscoverBaby:(id)baby
                          Peripheral:(CBPeripheral *)peripheral
                   AdvertisementData:(NSDictionary *)advertisementData
                                RSSI:(NSNumber *)RSSI;

/// 檢查MAC地址類型 / Check MAC address type
/// @return MAC地址類型 / MAC address type
/// - 0: 無MAC地址 / No MAC address
/// - 1: MAC地址格式正確 / MAC address format is correct
/// - 2: MAC地址格式錯誤 / MAC address format error
- (NSInteger)checkMacType;

/// 透過藍牙管理器連接設備 / Connect device through bluetooth manager
- (void)connectDevice;

/// 透過藍牙管理器斷開設備連接 / Disconnect device through bluetooth manager
- (void)disconnectDevice;

/// 透過藍牙管理器斷開所有設備連接 / Disconnect all device through bluetooth manager
- (void)cancelAllConnect;

/// 解析設備資訊為字典格式 / Parse device information to dictionary
- (NSMutableDictionary *)parseDictionary;

@end

NS_ASSUME_NONNULL_END
