//
//  MicroLifeCloud2GManager.h
//  MicroLifeDeviceSDK
//
//  Created by willy.wu on 2022/5/20.
//

#import <MicroLifeDeviceSDK/MicroLifeDeviceSDK.h>
#import <MicroLifeDeviceSDK/MicroLifeCloudDataModel.h>

typedef void(^CloudDataBlock) (bool success, NSInteger code, MicroLifeCloudDataModel * _Nullable dataModel, NSError * _Nullable error);
typedef void(^CloudDataFulfillBlock) (bool success, NSInteger code, bool fulfill, MicroLifeCloudDataModel * _Nullable dataModel, NSError * _Nullable error);

NS_ASSUME_NONNULL_BEGIN

/**
 SeverType

 - MicroLifeTestSever: Test Sever
 - MicroLifeProductSever: Product Sever
 - MicroLifeTestUSASever: Test USA Sever
 - MicroLifeProductUSASever: Product USA Sever
 - MicroLifeTestWalgreenSever: Test Walgreen Sever
 - MicroLifeProductWalgreenSever: Product Walgreen Sever
 */
typedef NS_ENUM(NSInteger, MicroLifeSeverType) {
    MicroLifeDeveloperSever,
    MicroLifeTestSever,
    MicroLifeProductSever,
    MicroLifeTestUSASever,
    MicroLifeProductUSASever
};

/**
 CloudDataType
 - CloudDataTypeBPM:
 - CloudDataTypeWeight:
 - CloudDataTypeEvent:
 - CloudDataTypeTemperature:
 - CloudDataTypeOxygen:
 */
typedef NS_ENUM(NSInteger , MicroLifeCloudDataType) {
    CloudDataTypeEvent,
    CloudDataTypeBPM,
    CloudDataTypeWeight,
    CloudDataTypeTemperature,
    CloudDataTypeOxygen,
    CloudDataTypePFM,
    CloudDataTypeDevice,
    CloudDataTypeEmail,
    CloudDataTypePushNotification,
    CloudDataTypeFAQ
};

/**
 MicroLifeCloudDataTimeType
 - CloudDataTimeModifyDate:
 - CloudDataTimeUpDate:
 */
typedef NS_ENUM(NSInteger , MicroLifeCloudDataTimeType) {
    CloudDataTimeModifyDate,
    CloudDataTimeUpDate
};

@interface MicroLifeCloud2GManager : CloudManager

@property (nonatomic, strong) NSString *redirect_url;

@property (nonatomic, strong) NSString *access_token;
@property (nonatomic, strong) NSString *refresh_token;
@property (nonatomic, strong) NSDate *tokenExpirationTime;

/// 伺服器類型屬性
/// Server type property
@property (nonatomic, assign) MicroLifeSeverType severType;

/**
 * 單例方法：用於初始化伺服器配置
 * Singleton method: Used to initialize server configuration
 *
 * @param severType 伺服器類型 (Server type)
 *                  - MicroLifeTestSever: 測試伺服器 (Test Server)
 *                  - MicroLifeProductSever: 生產伺服器 (Production Server)
 *                  - MicroLifeProductUSASever: 美國生產伺服器 (Product USA Server)
 * @param client_id 用戶端 ID (Client ID)
 * @param client_secret 用戶端密鑰 (Client Secret)
 * @param redirect_url 回應URL (redirect url)
 * @param showLog 是否顯示日誌 (Whether to show logs)
 * @return 返回單例對象 (Returns a singleton instance)
 */
+ (instancetype)shareWhithSeverType:(MicroLifeSeverType)severType
                                 ID:(NSString *)client_id
                             Secret:(NSString *)client_secret
                        RedirectURL:(NSString *)redirect_url
                            showLog:(BOOL)showLog;

/**
 * 設置主要伺服器類型
 * Set the main server type
 *
 * @param severType 伺服器類型 (Server type)
 *                  - MicroLifeTestSever: 測試伺服器 (Test Server)
 *                  - MicroLifeProductSever: 生產伺服器 (Production Server)
 *                  - MicroLifeProductUSASever: 美國生產伺服器 (Product USA Server)
 */
- (void)setMainSeverType:(MicroLifeSeverType)severType;

/**
 * 檢查區域設定
 * Check region configuration
 *
 * @param countryCode 國家代碼 (Country code)
 * @return 返回區域名稱 (Returns the region name)
 */
- (NSString *)checkRegion:(NSString *)countryCode;

/**
 * 設置區域
 * Set region
 *
 * @param countryCode 國家代碼 (Country code)
 */
- (void)setRegion:(NSString *)countryCode;

/**
 * 檢查當前語言
 * Check current language
 *
 * @param currentLanguage 當前語言代碼 (Current language code)
 * @return 返回語言名稱 (Returns the language name)
 */
- (NSString *)checkCurrentLanguage:(NSString *)currentLanguage;

/**
 * 設置當前語言
 * Set current language
 *
 * @param currentLanguage 當前語言代碼 (Current language code)
 */
- (void)setCurrentLanguage:(NSString *)currentLanguage;

/**
 * 設置是否顯示日誌
 * Set whether to show logs
 *
 * @param showLog 是否顯示日誌 (Whether to show logs)
 */
- (void)showLog:(BOOL)showLog;

/**
 * 清除用戶令牌
 * Clear user token
 */
- (void)clearedToken;

/**
 * 獲取主要 URL
 * Get the main URL
 *
 * @return 返回主要 URL (Returns the main URL)
 */
- (NSString *)mainURL;

/**
 * 用戶註冊(SFSafariViewController)
 * Create Account
 *
 * @param vc 當前視圖控制器 (Current view controller)
 * @param flag 是否動畫顯示 (Whether to animate display)
 * @param completion 完成回調 (Completion callback)
 */
- (void)createAccountWithVC:(id)vc animated:(BOOL)flag completion:(void (^)(void))completion;

/**
 * 用戶註冊(ASWebAuthenticationSession)
 * Create Account
 *
 * @param completionHandler 完成回調 (Completion callback)
 */
- (void)createAccount:(void(^)(NSString * _Nullable authCode, NSError * _Nullable error))completionHandler;

/**
 * 用戶登錄(SFSafariViewController)
 * User login
 *
 * @param vc 當前視圖控制器 (Current view controller)
 * @param flag 是否動畫顯示 (Whether to animate display)
 * @param completion 完成回調 (Completion callback)
 */
- (void)loginWithVC:(id)vc
           animated:(BOOL)flag
         completion:(void (^)(void))completion;

/**
 * 用戶登錄(ASWebAuthenticationSession)
 * User login
 *
 * @param completionHandler 完成回調 (Completion callback)
 */
- (void)login:(void(^)(NSString * _Nullable authCode, NSError * _Nullable error))completionHandler;

/**
 * 更改密碼(SFSafariViewController)
 * Change password
 *
 * @param vc 當前視圖控制器 (Current view controller)
 * @param flag 是否動畫顯示 (Whether to animate display)
 * @param completion 完成回調 (Completion callback)
 */
- (void)changePasswordWithVC:(id)vc
                    animated:(BOOL)flag
                  completion:(void (^)(void))completion;
/**
 * 更改密碼(ASWebAuthenticationSession)
 * Change password
 *
 * @param completionHandler 完成回調 (Completion callback)
 */
- (void)changePassword:(void(^)(NSURL * _Nullable callbackURL, NSError * _Nullable error))completionHandler;

/**
 * 掃描 BP (SFSafariViewController)
 * Scan BP
 *
 * @param vc 當前視圖控制器 (Current view controller)
 * @param flag 是否動畫顯示 (Whether to animate display)
 * @param completion 完成回調 (Completion callback)
 */
- (void)scanbpWithVC:(id)vc animated:(BOOL)flag completion:(void (^)(void))completion;

/**
 * 掃描 BP (ASWebAuthenticationSession)
 * Scan BP
 *
 * @param completionHandler 完成回調 (Completion callback)
 */
- (void)scanbp:(void(^)(NSURL * _Nullable callbackURL, NSError * _Nullable error))completionHandler;

/**
 * 提交 OAuth Token
 * Submit OAuth Token
 *
 * @param grant_type 授權類型 (Grant type)
 * @param code 授權碼 (Authorization code)
 * @param redirect_uri 重定向 URI (Redirect URI)
 * @param block 完成回調 (Completion callback)
 */
- (void)postOAuthToken:(GrantType)grant_type
                  code:(NSString *)code
           redirectURI:(NSString *)redirect_uri
           FinishBlock:(CloudDataBlock)block;

/**
 * 提交系統信息
 * Submit system information
 *
 * @param client_unique_id 客戶端唯一 ID (Client unique ID)
 * @param os 操作系統 (Operating system)
 * @param machine_type 機器類型 (Machine type)
 * @param push_unique_id 推送唯一 ID (Push unique ID)
 * @param model 設備型號 (Device model)
 * @param company 公司名稱 (Company name)
 * @param gps GPS 信息 (GPS information)
 * @param block 完成回調 (Completion callback)
 */
- (void)postSys:(NSString *)client_unique_id
             os:(NSString *)os
   machine_type:(NSString *)machine_type
 push_unique_id:(NSString *)push_unique_id
          model:(NSString *)model
        company:(NSString *)company
            gps:(NSString *)gps
    FinishBlock:(CloudDataBlock)block;

/**
 * 獲取用戶基本數據
 * Get user base data
 *
 * @param block 完成回調 (Completion callback)
 */
- (void)getMemberBaseData:(CloudDataBlock)block;

/**
 * 獲取用戶數據
 * Get user data
 *
 * @param block 完成回調 (Completion callback)
 */
- (void)getMemberData:(CloudDataBlock)block;


/**
 * 修改用戶資料
 * Modify user profile
 *
 * @param name 用戶名稱 (User name)
 * @param birthday 用戶生日 (User birthday)
 * @param gender 性別 (Gender)
 * @param height 身高 (Height)
 * @param weight 體重 (Weight)
 * @param unit_type 單位類型 (Unit type)
 * @param sys_unit 血壓單位 (Blood pressure unit)
 * @param sys 收縮壓 (Systolic pressure)
 * @param dia 舒張壓 (Diastolic pressure)
 * @param goal_weight 目標體重 (Goal weight)
 * @param body_fat 體脂率 (Body fat percentage)
 * @param bmi 身體質量指數 (BMI)
 * @param sys_activity 收縮壓活動狀態 (Systolic activity status)
 * @param dia_activity 舒張壓活動狀態 (Diastolic activity status)
 * @param weight_activity 體重活動狀態 (Weight activity status)
 * @param body_fat_activity 體脂活動狀態 (Body fat activity status)
 * @param bmi_activity BMI 活動狀態 (BMI activity status)
 * @param threshold 閾值 (Threshold)
 * @param cuff_size 血壓袖帶尺寸 (Cuff size)
 * @param bp_measurement_arm 測量手臂 (Measurement arm)
 * @param resistance 電阻 (Resistance)
 * @param measuringTime 測量時間 (Measuring time)
 * @param measuringTime_activity 測量時間活動狀態 (Measuring time activity status)
 * @param oxygen 血氧數值 (Oxygen level)
 * @param oxygen_activity 血氧活動狀態 (Oxygen activity status)
 * @param pef_activity 峰值流量活動狀態 (Peak flow activity status)
 * @param pef_best_value 峰值流量最佳值 (Peak flow best value)
 * @param pef_red_zone 峰值流量紅色區域 (Peak flow red zone)
 * @param date_format 日期格式 (Date format)
 * @param conditions 健康條件 (Health conditions)
 * @param is_sync 同步狀態 (is_sync)
 * @param photo 用戶照片 (User photo)
 * @param block 完成回調 (Completion callback)
 */
- (void)postModifyMemberWhithName:(NSString *)name
                         birthday:(NSString *)birthday
                           gender:(NSInteger)gender
                           height:(NSInteger)height
                           weight:(float)weight
                        unit_type:(NSInteger)unit_type
                         sys_unit:(NSInteger)sys_unit
                              sys:(float)sys
                              dia:(float)dia
                      goal_weight:(float)goal_weight
                         body_fat:(float)body_fat
                              bmi:(float)bmi
                     sys_activity:(BOOL)sys_activity
                     dia_activity:(BOOL)dia_activity
                  weight_activity:(BOOL)weight_activity
                body_fat_activity:(BOOL)body_fat_activity
                     bmi_activity:(BOOL)bmi_activity
                        threshold:(NSInteger)threshold
                        cuff_size:(NSInteger)cuff_size
               bp_measurement_arm:(NSInteger)bp_measurement_arm
                       resistance:(NSInteger)resistance
                    measuringTime:(NSInteger)measuringTime
           measuringTime_activity:(BOOL)measuringTime_activity
                           oxygen:(NSInteger)oxygen
                  oxygen_activity:(BOOL)oxygen_activity
                     pef_activity:(BOOL)pef_activity
                   pef_best_value:(NSInteger)pef_best_value
                     pef_red_zone:(NSInteger)pef_red_zone
                      date_format:(NSInteger)date_format
                       conditions:(NSString *)conditions
                          is_sync:(BOOL)is_sync
                            photo:(id _Nullable)photo
                      FinishBlock:(CloudDataBlock)block;

/**
 * 刪除用戶資料
 * Delete user profile
 *
 * @param block 完成回調 (Completion callback)
 */
- (void)postDeletetModifyMember:(CloudDataBlock)block;

/**
 * 獲取歷史數據
 * Get historical data
 *
 * @param dataType 數據類型 (Data type)
 *                 - CloudDataTypeEvent: 事件數據 (Event Data)
 *                 - CloudDataTypeBPM: 血壓數據 (Blood Pressure Data)
 *                 - CloudDataTypeWeight: 體重數據 (Weight Data)
 *                 - CloudDataTypeTemperature: 體溫數據 (Temperature Data)
 *                 - CloudDataTypeOxygen: 血氧數據 (Oxygen Data)
 *                 - CloudDataTypePFM: 峰值流量數據 (Peak Flow Data)
 * @param start 起始日期 (Start date)
 * @param end 結束日期 (End date)
 * @param limit 每次請求的數據上限 (Data limit per request)
 * @param offset 數據偏移量 (Data offset)
 * @param time_type 時間類型 (Time type)
 *                  - CloudDataTimeModifyDate: 修改時間 (Modification Date)
 *                  - CloudDataTimeUpDate: 更新時間 (Update Date)
 * @param automatic 是否自動加載更多數據 (Whether to automatically load more data)
 * @param block 完成回調 (Completion callback)
 */
- (void)getHistoryDataWhithType:(MicroLifeCloudDataType)dataType
                          Start:(NSString *)start
                            End:(NSString *)end
                          limit:(NSInteger)limit
                         offset:(NSInteger)offset
                       timeType:(MicroLifeCloudDataTimeType)time_type
                      automatic:(BOOL)automatic
                    FinishBlock:(CloudDataFulfillBlock)block;

/**
 * 方法：提交血壓測量數據到伺服器或雲端系統
 * Method: Submit blood pressure measurement data to the server or cloud system
 *
 * 參數說明 (Parameter Description):
 * @param user_id      使用者的唯一識別碼 (User's unique identifier) (NSString)
 * @param sys          收縮壓 (Systolic Pressure)，浮點數 (float)
 * @param dia          舒張壓 (Diastolic Pressure)，浮點數 (float)
 * @param pul          脈搏 (Pulse)，整數 (NSInteger)
 * @param cuffokr      血壓袖帶是否正確佩戴 (Whether the cuff is properly worn)，
 *                     0 表示不正確 (0: Incorrect)，1 表示正確 (1: Correct) (NSInteger)
 * @param bpm_id       血壓計的唯一識別碼 (Blood pressure monitor's unique identifier) (NSString)
 * @param afib         是否檢測到心房顫動 (Whether atrial fibrillation is detected)，
 *                     0 表示無 (0: No)，1 表示有 (1: Yes) (NSInteger)
 * @param pad          是否檢測到脈搏異常 (Whether pulse arrhythmia is detected)，
 *                     0 表示正常 (0: Normal)，1 表示異常 (1: Abnormal) (NSInteger)
 * @param man          測量模式 (Measurement mode)，
 *                     3G 偵測模式 (3G detection mode):
 *                     0: BP_Single_MOD（單次測量模式 / Single measurement mode）
 *                     3: BP_MAM_MODE + SW AFib ON（MAM 模式 + 軟體 AFib 開啟 / MAM mode + Software AFib ON）
 *                     4G 偵測模式 (4G detection mode):
 *                     0: BP_Single_MOD（單次測量模式 / Single measurement mode）
 *                     1: BP_Single_MODE + SW AFib ON（單次測量模式 + 軟體 AFib 開啟 / Single mode + Software AFib ON）
 *                     2: BP_MAM_MODE（MAM 模式 / MAM mode）
 *                     3: BP_MAM_MODE + SW AFib ON（MAM 模式 + 軟體 AFib 開啟 / MAM mode + Software AFib ON）(NSInteger)
 * @param algorithm    使用的演算法類型 (Type of algorithm used)，整數 (NSInteger)
 *                     0: Consumer Mode（消費者模式 / Consumer mode）
 *                     1: Usual Mode（普通模式 / Usual mode）
 *                     2: Diagnostic Mode（診斷模式 / Diagnostic mode）
 *                     3: ABPM Mode（動態血壓監測模式 / Ambulatory Blood Pressure Monitoring mode）
 * @param source       數據來源 (Data source)，整數 (NSInteger)
 *                     0: APP Manual Input（APP 手動輸入 / APP manual input）
 *                     1: APP BT Device Input（APP 藍牙設備輸入 / APP Bluetooth device input）
 *                     2: Web Manual Input（網頁手動輸入 / Web manual input）
 *                     3: APP NFC Device Input（APP NFC 設備輸入 / APP NFC device input）
 *                     4: Scan BP Input（掃描血壓數據輸入 / Scan blood pressure data input）
 *                     5: Cellular BPM（蜂窩網絡血壓計輸入 / Cellular blood pressure monitor input）
 * @param date         測量的日期和時間 (Measurement date and time) (NSString)
 * @param mac_address  設備的 MAC 地址 (Device's MAC address)，
 *                     用於識別藍牙或網路設備 (Used to identify Bluetooth or network devices) (NSString)
 * @param block  操作完成後的回呼區塊 (Callback block after operation is completed)，
 *                     用於處理伺服器回應或結果 (Used to handle server responses or results) (CloudDataBlock)
 */
- (void)postAddBPMWhithID:(NSString *)user_id
                      sys:(float)sys
                      dia:(float)dia
                      pul:(NSInteger)pul
                  cuffokr:(NSInteger)cuffokr
                   bpm_id:(NSString *)bpm_id
                     afib:(NSInteger)afib
                      pad:(NSInteger)pad
                      man:(NSInteger)man
                algorithm:(NSInteger)algorithm
                   source:(NSInteger)source
                     date:(NSString *)date
              mac_address:(NSString *)mac_address
              FinishBlock:(CloudDataBlock)block;


/**
 * 編輯血壓測量數據
 * Edit blood pressure measurement data
 *
 * @param bpm_id 血壓數據 ID (Blood pressure data ID)
 * @param sys 收縮壓 (Systolic pressure)
 * @param dia 舒張壓 (Diastolic pressure)
 * @param pul 脈搏 (Pulse)
 * @param date 測量日期 (Measurement date)
 * @param block 完成回調 (Completion callback)
 */
- (void)postEditBPMWhithID:(NSString *)bpm_id
                       sys:(float)sys
                       dia:(float)dia
                       pul:(NSInteger)pul
                      date:(NSString *)date
               FinishBlock:(CloudDataBlock)block;

/**
 * 添加體重測量數據
 * Add weight measurement data
 *
 * @param weight_id 體重數據 ID (Weight data ID)
 * @param weight 體重 (Weight)
 * @param bmi 身體質量指數 (BMI)
 * @param body_fat 體脂率 (Body fat percentage)
 * @param water 身體水分 (Body water percentage)
 * @param skeleton 骨骼重量 (Skeletal weight)
 * @param muscle 肌肉重量 (Muscle weight)
 * @param bmr 基礎代謝率 (Basal metabolic rate)
 * @param organ_fat 內臟脂肪 (Visceral fat)
 * @param date 測量日期 (Measurement date)
 * @param mac_address 設備 MAC 地址 (Device MAC address)
 * @param block 完成回調 (Completion callback)
 */
- (void)postAddWeightDataWhithID:(NSString *)weight_id
                          weight:(float)weight
                             bmi:(float)bmi
                        body_fat:(float)body_fat
                           water:(NSInteger)water
                        skeleton:(float)skeleton
                          muscle:(NSInteger)muscle
                             bmr:(NSInteger)bmr
                       organ_fat:(float)organ_fat
                            date:(NSString *)date
                     mac_address:(NSString *)mac_address
                     FinishBlock:(CloudDataBlock)block;

/**
 * 編輯體重測量數據
 * Edit weight measurement data
 *
 * @param weight_id 體重數據 ID (Weight data ID)
 * @param weight 體重 (Weight)
 * @param bmi 身體質量指數 (BMI)
 * @param body_fat 體脂率 (Body fat percentage)
 * @param date 測量日期 (Measurement date)
 * @param block 完成回調 (Completion callback)
 */
- (void)postEditWeightDataWhithID:(NSString *)weight_id
                           weight:(float)weight
                              bmi:(float)bmi
                         body_fat:(float)body_fat
                             date:(NSString *)date
                      FinishBlock:(CloudDataBlock)block;

/**
 * 添加體溫事件數據
 * Add Bluetooth event data
 *
 * @param event_code 事件代碼 (Event code)
 * @param event 事件描述 (Event description)
 * @param type 事件類型 (Event type)
 * @param event_time 事件時間 (Event time)
 * @param block 完成回調 (Completion callback)
 */
- (void)postAddBTEventWhithCode:(NSInteger)event_code
                          event:(NSString *)event
                           type:(NSString *)type
                     event_time:(NSString *)event_time
                    FinishBlock:(CloudDataBlock)block;

/**
 * 編輯體溫事件數據
 * Edit Bluetooth event data
 *
 * @param event_code 事件代碼 (Event code)
 * @param event 事件描述 (Event description)
 * @param type 事件類型 (Event type)
 * @param event_time 事件時間 (Event time)
 * @param isDelete 是否刪除該事件 (Whether to delete the event)
 * @param block 完成回調 (Completion callback)
 */
- (void)postEditBTEventWhithCode:(NSInteger)event_code
                           event:(NSString *)event
                            type:(NSString *)type
                      event_time:(NSString *)event_time
                        isDelete:(BOOL)isDelete
                     FinishBlock:(CloudDataBlock)block;

/**
 * 添加體溫測量數據
 * Add Temperature measurement data
 *
 * @param event_code 事件代碼 (Event code)
 * @param bt_id 體溫 ID (Temperature ID)
 * @param body_temp 身體溫度 (Body temperature)
 * @param room_temp 室溫 (Room temperature)
 * @param date 測量日期 (Measurement date)
 * @param mac_address 設備 MAC 地址 (Device MAC address)
 * @param block 完成回調 (Completion callback)
 */
- (void)postAddBTDataWhithCode:(NSInteger)event_code
                         bt_id:(NSString *)bt_id
                     body_temp:(float)body_temp
                     room_temp:(float)room_temp
                          date:(NSString *)date
                   mac_address:(NSString *)mac_address
                   FinishBlock:(CloudDataBlock)block;

/**
 * 編輯體溫測量數據
 * Edit Temperature measurement data
 *
 * @param event_code 事件代碼 (Event code)
 * @param bt_id 體溫 ID (Temperature ID)
 * @param body_temp 身體溫度 (Body temperature)
 * @param room_temp 室溫 (Room temperature)
 * @param date 測量日期 (Measurement date)
 * @param block 完成回調 (Completion callback)
 */
- (void)postEditBTDataWhithCode:(NSInteger)event_code
                          bt_id:(NSString *)bt_id
                      body_temp:(float)body_temp
                      room_temp:(float)room_temp
                           date:(NSString *)date
                    FinishBlock:(CloudDataBlock)block;

/**
 * 添加血氧儀測量數據
 * Add oximeter measurement data
 *
 * @param oxymeter_id 血氧儀數據 ID (Oximeter data ID)
 * @param spo2 血氧飽和度 (SpO2 level)
 * @param pulse_rate 脈搏速率 (Pulse rate)
 * @param perfusion_index 灌注指數 (Perfusion index)
 * @param spo2_raw_data 血氧原始數據 (SpO2 raw data)
 * @param pulse_rate_raw_data 脈搏速率原始數據 (Pulse rate raw data)
 * @param perfusion_index_raw_data 灌注指數原始數據 (Perfusion index raw data)
 * @param spo2_waveform_data 血氧波形數據 (SpO2 waveform data)
 * @param measurement_length 測量時間長度 (Measurement length)
 * @param date 測量日期 (Measurement date)
 * @param mac_address 設備 MAC 地址 (Device MAC address)
 * @param block 完成回調 (Completion callback)
 */
- (void)postAddOxymeterDataWhithID:(NSString *)oxymeter_id
                              spo2:(float)spo2
                        pulse_rate:(float)pulse_rate
                   perfusion_index:(float)perfusion_index
                     spo2_raw_data:(NSString *)spo2_raw_data
               pulse_rate_raw_data:(NSString *)pulse_rate_raw_data
          perfusion_index_raw_data:(NSString *)perfusion_index_raw_data
                spo2_waveform_data:(NSString *)spo2_waveform_data
                measurement_length:(NSInteger)measurement_length
                              date:(NSString *)date
                       mac_address:(NSString *)mac_address
                       FinishBlock:(CloudDataBlock)block;

/**
 * 編輯血氧儀測量數據
 * Edit oximeter measurement data
 *
 * @param oxymeter_id 血氧儀數據 ID (Oximeter data ID)
 * @param spo2 血氧飽和度 (SpO2 level)
 * @param pulse_rate 脈搏速率 (Pulse rate)
 * @param perfusion_index 灌注指數 (Perfusion index)
 * @param measurement_length 測量時間長度 (Measurement length)
 * @param date 測量日期 (Measurement date)
 * @param block 完成回調 (Completion callback)
 */
- (void)postEditOxymeterDataWhithID:(NSString *)oxymeter_id
                               spo2:(float)spo2
                         pulse_rate:(float)pulse_rate
                    perfusion_index:(float)perfusion_index
                 measurement_length:(NSInteger)measurement_length
                               date:(NSString *)date
                        FinishBlock:(CloudDataBlock)block;

/**
 * 添加峰值流量測量數據
 * Add peak flow measurement data
 *
 * @param peakflow_id 峰值流量數據 ID (Peak flow data ID)
 * @param pef 峰值呼氣流速 (Peak expiratory flow, PEF)
 * @param fev1 第一秒用力呼氣容積 (Forced expiratory volume in 1 second, FEV1)
 * @param date 測量日期 (Measurement date)
 * @param mac_address 設備 MAC 地址 (Device MAC address)
 * @param block 完成回調 (Completion callback)
 */
- (void)postAddPeakFlowDataWhithID:(NSString *)peakflow_id
                               pef:(NSInteger)pef
                              fev1:(float)fev1
                              date:(NSString *)date
                       mac_address:(NSString *)mac_address
                       FinishBlock:(CloudDataBlock)block;

/**
 * 編輯峰值流量測量數據
 * Edit peak flow measurement data
 *
 * @param peakflow_id 峰值流量數據 ID (Peak flow data ID)
 * @param pef 峰值呼氣流速 (Peak expiratory flow, PEF)
 * @param fev1 第一秒用力呼氣容積 (Forced expiratory volume in 1 second, FEV1)
 * @param date 測量日期 (Measurement date)
 * @param block 完成回調 (Completion callback)
 */
- (void)postEditPeakFlowDataWhithID:(NSString *)peakflow_id
                                pef:(NSInteger)pef
                               fev1:(float)fev1
                               date:(NSString *)date
                        FinishBlock:(CloudDataBlock)block;

/**
 * 添加記錄數據
 * Add record data
 *
 * @param type_id 數據類型 ID (Data type ID)
 * @param note_type 記錄類型 (Record type)
 *                 - 1: 血壓 (BPM)
 *                 - 2: 體重 (Weight)
 *                 - 3: 體溫 (Temperature)
 *                 - 4: 血氧 (Oxygen)
 *                 - 5: 峰值流量 (PFM)
 * @param note 記錄內容 (Note content)
 * @param photo 附加照片 (Attached photo)
 * @param recording 附加錄音 (Attached recording)
 * @param recording_time 錄音時長 (Recording duration)
 * @param block 完成回調 (Completion callback)
 */
- (void)postAddNoteDataWhithID:(NSString *)type_id
                     note_type:(NSInteger)note_type
                          note:(NSString *)note
                         photo:(id _Nullable)photo
                     recording:(NSData * _Nullable)recording
                recording_time:(NSString * _Nullable)recording_time
                   FinishBlock:(CloudDataBlock)block;

/**
 * 刪除記錄數據
 * Delete record data
 *
 * @param type_id 數據類型 ID (Data type ID)
 * @param note_type 記錄類型 (Record type)
 *                 - 1: 血壓 (BPM)
 *                 - 2: 體重 (Weight)
 *                 - 3: 體溫 (Temperature)
 *                 - 4: 血氧 (Oxygen)
 *                 - 5: 峰值流量 (PFM)
 * @param block 完成回調 (Completion callback)
 */
- (void)postDeleteRecordDataWhithID:(NSString *)type_id
                          note_type:(NSString *)note_type
                        FinishBlock:(CloudDataBlock)block;

/**
 * 下載血壓測量數據的 PDF 報告
 * Download blood pressure measurement data as a PDF report
 *
 * @param start_date 報告的開始日期 (Start date of the report)
 * @param end_date 報告的結束日期 (End date of the report)
 * @param sys_threshold 收縮壓閾值 (Systolic pressure threshold)
 * @param dia_threshold 舒張壓閾值 (Diastolic pressure threshold)
 * @param unit 單位 (Unit)
 * @param decimal 小數位數 (Decimal places)
 * @param show_arrhythmia 是否顯示心律不整 (Whether to show arrhythmia)
 * @param date_format 日期格式 (Date format)
 * @param photo 附加照片 (Attached photo)
 * @param block 完成回調 (Completion callback)
 */
- (void)postDownloadBPMPDF:(NSString *)start_date
                  end_date:(NSString *)end_date
             sys_threshold:(NSInteger)sys_threshold
             dia_threshold:(NSInteger)dia_threshold
                      unit:(NSInteger)unit
                   decimal:(NSInteger)decimal
           show_arrhythmia:(NSInteger)show_arrhythmia
               date_format:(NSInteger)date_format
                     photo:(id)photo
               FinishBlock:(CloudDataBlock)block;

/**
 * 下載藍牙測量數據的 PDF 報告
 * Download Bluetooth measurement data as a PDF report
 *
 * @param start_date 報告的開始日期 (Start date of the report)
 * @param end_date 報告的結束日期 (End date of the report)
 * @param event_code 事件代碼 (Event code)
 * @param threshold 閾值 (Threshold)
 * @param unit 單位 (Unit)
 * @param decimal 小數位數 (Decimal places)
 * @param date_format 日期格式 (Date format)
 * @param photo 附加照片 (Attached photo)
 * @param block 完成回調 (Completion callback)
 */
- (void)postDownloadBTPDF:(NSString *)start_date
                 end_date:(NSString *)end_date
                eventCode:(NSInteger)event_code
                threshold:(float)threshold
                     unit:(NSInteger)unit
                  decimal:(NSInteger)decimal
              date_format:(NSInteger)date_format
                    photo:(id)photo
              FinishBlock:(CloudDataBlock)block;

/**
 * 下載體重測量數據的 PDF 報告
 * Download weight measurement data as a PDF report
 *
 * @param start_date 報告的開始日期 (Start date of the report)
 * @param end_date 報告的結束日期 (End date of the report)
 * @param wei_threshold 體重閾值 (Weight threshold)
 * @param bmi_threshold BMI 閾值 (BMI threshold)
 * @param fat_threshold 體脂率閾值 (Body fat percentage threshold)
 * @param unit 單位 (Unit)
 * @param decimal 小數位數 (Decimal places)
 * @param show_info 是否顯示附加信息 (Whether to show additional information)
 * @param date_format 日期格式 (Date format)
 * @param photo 附加照片 (Attached photo)
 * @param block 完成回調 (Completion callback)
 */
- (void)postDownloadWeightPDF:(NSString *)start_date
                     end_date:(NSString *)end_date
                wei_threshold:(float)wei_threshold
                bmi_threshold:(float)bmi_threshold
                fat_threshold:(float)fat_threshold
                         unit:(NSInteger)unit
                      decimal:(NSInteger)decimal
                    show_info:(NSInteger)show_info
                  date_format:(NSInteger)date_format
                        photo:(id)photo
                  FinishBlock:(CloudDataBlock)block;

/**
 * 下載血氧測量數據的 PDF 報告
 * Download oximeter measurement data as a PDF report
 *
 * @param start_date 報告的開始日期 (Start date of the report)
 * @param end_date 報告的結束日期 (End date of the report)
 * @param threshold 閾值 (Threshold)
 * @param unit 單位 (Unit)
 * @param decimal 小數位數 (Decimal places)
 * @param date_format 日期格式 (Date format)
 * @param photo 附加照片 (Attached photo)
 * @param block 完成回調 (Completion callback)
 */
- (void)postDownloadOxymeterPDF:(NSString *)start_date
                       end_date:(NSString *)end_date
                      threshold:(float)threshold
                           unit:(NSInteger)unit
                        decimal:(NSInteger)decimal
                    date_format:(NSInteger)date_format
                          photo:(id)photo
                    FinishBlock:(CloudDataBlock)block;

/**
 * 下載峰值流量測量數據的 PDF 報告
 * Download peak flow measurement data as a PDF report
 *
 * @param start_date 報告的開始日期 (Start date of the report)
 * @param end_date 報告的結束日期 (End date of the report)
 * @param decimal 小數位數 (Decimal places)
 * @param date_format 日期格式 (Date format)
 * @param photo 附加照片 (Attached photo)
 * @param block 完成回調 (Completion callback)
 */
- (void)postDownloadPeakflowPDF:(NSString *)start_date
                       end_date:(NSString *)end_date
                        decimal:(NSInteger)decimal
                    date_format:(NSInteger)date_format
                          photo:(id)photo
                    FinishBlock:(CloudDataBlock)block;

/**
 * 添加設備
 * Add a device
 *
 * @param mac_address 設備 MAC 地址 (Device MAC address)
 * @param device_type 設備類型 (Device type)
 * @param bt_code 設備型號 (bt code)
 * @param device_model 設備型號 (Device model)
 * @param error_code 錯誤代碼 (Error code)
 * @param fw_version 固件版本 (Firmware version)
 * @param block 完成回調 (Completion callback)
 */
- (void)postAddDevice:(NSString *)mac_address device_type:(NSInteger)device_type bt_code:(NSInteger)bt_code device_model:(NSString *)device_model error_code:(NSString *)error_code fw_version:(NSString *)fw_version FinishBlock:(CloudDataBlock)bloc;

/**
 * 刪除設備數據
 * Delete device data
 *
 * @param mac_address 設備 MAC 地址 (Device MAC address)
 * @param block 完成回調 (Completion callback)
 */
- (void)postDeleteDeviceData:(NSString *)mac_address
                 FinishBlock:(CloudDataBlock)block;

/**
 * 添加郵件通知
 * Add email notification
 *
 * @param name 使用者名稱 (User name)
 * @param email 郵件地址 (Email address)
 * @param block 完成回調 (Completion callback)
 */
- (void)postAddMailNotification:(NSString *)name
                          email:(NSString *)email
                    FinishBlock:(CloudDataBlock)block;

/**
 * 編輯或刪除郵件通知
 * Edit or delete email notification
 *
 * @param mail_id 郵件 ID 列表 (List of mail IDs)
 * @param name 使用者名稱 (User name)
 * @param email 郵件地址 (Email address)
 * @param isDelete 是否刪除 (Whether to delete)
 * @param block 完成回調 (Completion callback)
 */
- (void)postEditMailNotification:(NSMutableArray<NSNumber *> *)mail_id
                            name:(NSString *)name
                           email:(NSString *)email
                        isDelete:(BOOL)isDelete
                     FinishBlock:(CloudDataBlock)block;

/**
 * 上傳 CSV 文件
 * Upload a CSV file
 *
 * @param csv CSV 文件數據 (CSV file data)
 * @param block 完成回調 (Completion callback)
 */
- (void)uploadCSV:(NSData *)csv
      FinishBlock:(CloudDataBlock)block;

/**
 * 檢查同步狀態
 * Check synchronization status
 *
 * @param import_id 導入數據 ID (Import data ID)
 * @param block 完成回調 (Completion callback)
 */
- (void)inspectionSyncStatus:(NSString *)import_id
                 FinishBlock:(CloudDataBlock)block;

/**
 * 提交聯絡支援的請求
 * Submit a contact support request
 *
 * @param operation_mode 操作模式（0: 雲端模式，1: 手機模式）
 *        Operation mode (0: cloud mode, 1: phone mode)
 * @param contact_name 聯絡人名稱，用於標明提交者的身份
 *        Contact name, used to identify the submitter
 * @param contact_email 聯絡人信箱，用於接收回應或進一步聯絡
 *        Contact email, used to receive responses or further communication
 * @param type 問題類型（0: App, 1: BT_Connection, 2: Consumer_BPM, 3: WatchBP_BPM, 4: Weight_Fat_Scale, 5: Thermometer, 6: Oximeter, 7: Peak_Flow_Meter, 8: Others）
 *        Issue type (0: App, 1: BT_Connection, 2: Consumer_BPM, 3: WatchBP_BPM, 4: Weight_Fat_Scale, 5: Thermometer, 6: Oximeter, 7: Peak_Flow_Meter, 8: Others)
 * @param description 問題描述，詳細說明需要支援的內容
 *        Issue description, a detailed explanation of the support needed
 * @param app_name APP 名稱，用於標明提交者使用的應用程式
 *        App name, used to identify the application being used
 * @param app_version APP 版本號，用於標明應用程式的版本
 *        App version, used to specify the version of the application
 * @param sdk_version SDK 版本號，用於標明應用程式中使用的 SDK 版本
 *        SDK version, used to specify the SDK version in the application
 * @param phone_model 手機裝置型號，例如 iPhone 14
 *        Phone model, e.g., iPhone 14
 * @param phone_os_version 手機裝置系統版本，例如 iOS 17.0
 *        Phone OS version, e.g., iOS 17.0
 * @param device_bt_model_name（選填）醫療裝置名稱，例如血壓計、體重計等裝置的名稱
 *        (Optional) Medical device name, e.g., blood pressure monitor, weight scale, etc.
 * @param device_sn（選填）裝置序號，用於唯一標識裝置
 *        (Optional) Device serial number, used to uniquely identify the device
 * @param device_bt_protocol_code（選填）裝置藍牙協議代碼
 *        (Optional) Device Bluetooth protocol code
 * @param device_bt_protocol_version（選填）裝置藍牙協議版本
 *        (Optional) Device Bluetooth protocol version
 * @param device_battery_voltage（選填）裝置電池電壓，範圍為 999.999 ~ -999.999
 *        (Optional) Device battery voltage, range: 999.999 ~ -999.999
 * @param device_fw_version（選填）裝置韌體版本
 *        (Optional) Device firmware version
 * @param device_fw_date（選填）裝置韌體日期
 *        (Optional) Device firmware date
 * @param language 語系，用於標明使用者的語言（例如 en、zh-TW）
 *        Language, used to specify the user's language (e.g., en, zh-TW)
 * @param locate 地區，用於標明使用者的地區（例如 TW、US）
 *        Location, used to specify the user's region (e.g., TW, US)
 * @param file（選填）記錄檔上傳，例如日誌檔案，用於提供更多的問題背景資訊
 *        (Optional) File upload, such as log files, to provide more background information about the issue
 * @param block 完成後的回調，用於處理 API 響應
 *        Callback after completion, used to handle API responses
 */
- (void)contactSupport:(NSInteger)operation_mode
            ontactName:(NSString *)contact_name
          ContactEmail:(NSString *)contact_email
                  Type:(NSInteger)type
           Description:(NSString *)description
               AppName:(NSString *)app_name
            AppVersion:(NSString *)app_version
            SDKVersion:(NSString *)sdk_version
            PhoneModel:(NSString *)phone_model
        PhoneOSVersion:(NSString *)phone_os_version
     DeviceBTModelName:(NSString *)device_bt_model_name
              DeviceSN:(NSString *)device_sn
  DeviceBTProtocolCode:(NSInteger)device_bt_protocol_code
DeviceBTProtocolVersion:(NSString *)device_bt_protocol_version
  DeviceBatteryVoltage:(NSString *)device_battery_voltage
       DeviceFWVersion:(NSString *)device_fw_version
          DeviceFWDate:(NSString *)device_fw_date
              Language:(NSString *)language
                Locate:(NSString *)locate
                  File:(id)file
           FinishBlock:(CloudDataBlock)block;

/**
 * 添加連線日誌
 * Add connection log
 *
 * @param account 帳戶名稱 (Account name)
 * @param display_name 顯示名稱 (Display name)
 * @param model_name 設備型號 (Device model name)
 * @param mac_address 設備 MAC 地址 (Device MAC address)
 * @param fw_version 固件版本 (Firmware version)
 * @param notes 備註 (Notes)
 * @param block 完成回調 (Completion callback)
 */
- (void)connectionLog:(NSString *)account
                 Name:(NSString *)display_name
                Model:(NSString *)model_name
                  Mac:(NSString *)mac_address
            FWVersion:(NSString *)fw_version
                 Note:(NSString *)notes
         FinishBlock:(CloudDataBlock)block;

/**
 * 上傳 ECG 原始數據
 * Upload ECG raw data
 *
 * @param name 檔案名稱 (File name)
 * @param file 檔案數據 (File data)
 * @param block 完成回調 (Completion callback)
 */
- (void)uploadECGRawdata:(NSString *)name
                    File:(id)file
             FinishBlock:(CloudDataBlock)block;

/**
 * 上傳 ECG PDF 檔案
 * Upload ECG PDF file
 *
 * @param name 檔案名稱 (File name)
 * @param file PDF 檔案數據 (PDF file data)
 * @param block 完成回調 (Completion callback)
 */
- (void)uploadECGPDF:(NSString *)name
                File:(id)file
         FinishBlock:(CloudDataBlock)block;
@end

NS_ASSUME_NONNULL_END
