//
//  BluetoothCallbacks.h
//  MicroLifeDeviceSDK
//
//  藍牙操作回調定義集合 (Bluetooth Operation Callbacks Collection)
//  用於定義所有與藍牙相關的操作回調，實現模塊化設計
//

#import <Foundation/Foundation.h>
#import <CoreBluetooth/CoreBluetooth.h>

NS_ASSUME_NONNULL_BEGIN

#pragma mark - 藍牙管理器狀態回調 (Bluetooth Manager State Callback)
/**
 * 藍牙管理器狀態更新回調
 * Bluetooth manager state update callback
 * @param state 藍牙管理器當前狀態 (current state of bluetooth manager)
 *             - CBManagerStateUnknown = 0      未知狀態 (Unknown)
 *             - CBManagerStateResetting = 1    重置中 (Resetting)
 *             - CBManagerStateUnsupported = 2  不支持 (Unsupported)
 *             - CBManagerStateUnauthorized = 3 未授權 (Unauthorized)
 *             - CBManagerStatePoweredOff = 4   已關閉 (Powered Off)
 *             - CBManagerStatePoweredOn = 5    已開啟 (Powered On)
 */
typedef void(^didUpdateStateBlock)(CBManagerState state) NS_SWIFT_NAME(DidUpdateStateBlock);

#pragma mark - 設備掃描回調 (Device Scanning Callbacks)
/**
 * 掃描到設備的回調
 * Device discovered callback
 * @param checkMacType MAC地址類型 (MAC address type)
 * @param device 掃描到的設備對象 (discovered device object <BLEDeviceInfo,MicroLifeDevice>)
 */
typedef void(^scanDeviceBlock)(NSInteger checkMacType, id device) NS_SWIFT_NAME(ScanDeviceBlock);

/**
 * 停止掃描回調
 * Cancel scanning callback
 */
typedef void(^cancelScanBlock)(void) NS_SWIFT_NAME(CancelScanBlock);

#pragma mark - 設備連接回調 (Device Connection Callbacks)
/**
 * 設備連接狀態變化回調
 * Device connection state change callback
 * @param device 待連接的設備 (device to connect <BLEDeviceInfo,MicroLifeDevice>)
 * @param central 藍牙中心管理器 (bluetooth central manager)
 * @param peripheral 藍牙外設 (bluetooth peripheral)
 * @param state 連接狀態 (connection state)
 * @param error 錯誤信息 (error information)
 */
typedef void(^connectDeviceStateBlock)(id device,
                                       CBCentralManager *central,
                                       CBPeripheral *peripheral,
                                       CBPeripheralState state,
                                       NSError * _Nullable error) NS_SWIFT_NAME(ConnectDeviceStateBlock);

/**
 * 斷開所有設備連接回調
 * Disconnect all devices callback
 */
typedef void(^cancelAllDevicesConnectionBlock)(void) NS_SWIFT_NAME(CancelAllDevicesConnectionBlock);

#pragma mark - 服務與特徵值發現回調 (Service and Characteristic Discovery Callbacks)
/**
 * 發現服務回調
 * Service discovery callback
 * @param device 設備對象 (device object <BLEDeviceInfo,MicroLifeDevice>)
 * @param peripheral 藍牙外設 (bluetooth peripheral)
 * @param error 錯誤信息 (error information)
 */
typedef void(^discoverServicesBlock)(id device,
                                     CBPeripheral *peripheral,
                                     NSError * _Nullable error) NS_SWIFT_NAME(DiscoverServicesBlock);

/**
 * 發現特徵值回調
 * Characteristic discovery callback
 * @param device 設備對象 (device object <BLEDeviceInfo,MicroLifeDevice>)
 * @param peripheral 藍牙外設 (bluetooth peripheral)
 * @param service 所屬服務 (belonged service)
 * @param error 錯誤信息 (error information)
 */
typedef void(^discoverCharacteristicsBlock)(id device,
                                            CBPeripheral *peripheral,
                                            CBService *service,
                                            NSError * _Nullable error) NS_SWIFT_NAME(DiscoverCharacteristicsBlock);

/**
 * 發現特徵值描述符回調
 * Characteristic descriptor discovery callback
 * @param device 設備對象 (device object <BLEDeviceInfo,MicroLifeDevice>)
 * @param peripheral 藍牙外設 (bluetooth peripheral)
 * @param characteristics 特徵值 (characteristic)
 * @param error 錯誤信息 (error information)
 */
typedef void(^discoverDescriptorsForCharacteristicBlock)(id device,
                                                         CBPeripheral *peripheral,
                                                         CBCharacteristic *characteristics,
                                                         NSError * _Nullable error) NS_SWIFT_NAME(DiscoverDescriptorsForCharacteristicBlock);

#pragma mark - 特徵值操作回調 (Characteristic Operation Callbacks)
/**
 * 特徵值通知狀態更新回調
 * Characteristic notification state update callback
 * @param characteristic 特徵值 (characteristic)
 * @param error 錯誤信息 (error information)
 */
typedef void(^updateNotificationStateForCharacteristicBlock)(CBCharacteristic *characteristic,
                                                             NSError * _Nullable error) NS_SWIFT_NAME(UpdateNotificationStateForCharacteristicBlock);

/**
 * 寫入特徵值回調
 * Write characteristic value callback
 * @param characteristic 特徵值 (characteristic)
 * @param error 錯誤信息 (error information)
 */
typedef void(^writeValueBlock)(CBCharacteristic *characteristic,
                               NSError * _Nullable error) NS_SWIFT_NAME(WriteValueBlock);

/**
 * 讀取特徵值回調
 * Read characteristic value callback
 * @param device 設備對象 (device object <BLEDeviceInfo,MicroLifeDevice>)
 * @param peripheral 藍牙外設 (bluetooth peripheral)
 * @param characteristics 特徵值 (characteristic)
 * @param error 錯誤信息 (error information)
 */
typedef void(^readValueBlock)(id device,
                              CBPeripheral *peripheral,
                              CBCharacteristic *characteristics,
                              NSError * _Nullable error) NS_SWIFT_NAME(ReadValueBlock);

/**
 * 接收特徵值通知回調
 * Receive characteristic notification callback
 * @param device 設備對象 (device object <BLEDeviceInfo,MicroLifeDevice>)
 * @param peripheral 藍牙外設 (bluetooth peripheral)
 * @param characteristics 特徵值 (characteristic)
 * @param error 錯誤信息 (error information)
 */
typedef void(^readNotifyValueForCharacteristicBlock)(id device,
                                                     CBPeripheral *peripheral,
                                                     CBCharacteristic *characteristics,
                                                     NSError * _Nullable error) NS_SWIFT_NAME(ReadNotifyValueForCharacteristicBlock);

#pragma mark - 描述符操作回調 (Descriptor Operation Callbacks)
/**
 * 讀取描述符值回調
 * Read descriptor value callback
 * @param device 設備對象 (device object <BLEDeviceInfo,MicroLifeDevice>)
 * @param peripheral 藍牙外設 (bluetooth peripheral)
 * @param descriptor 描述符 (descriptor)
 * @param error 錯誤信息 (error information)
 */
typedef void(^readValueForDescriptorsBlock)(id device,
                                            CBPeripheral *peripheral,
                                            CBDescriptor *descriptor,
                                            NSError * _Nullable error) NS_SWIFT_NAME(ReadValueForDescriptorsBlock);

#pragma mark - 信號強度回調 (RSSI Callback)
/**
 * 讀取信號強度回調
 * Read RSSI value callback
 * @param RSSI 信號強度值 (RSSI value)
 * @param error 錯誤信息 (error information)
 */
typedef void(^readRSSIBlock)(NSNumber *RSSI,
                             NSError * _Nullable error) NS_SWIFT_NAME(ReadRSSIBlock);

NS_ASSUME_NONNULL_END