//
//  BluetoothManagerProtocol.h
//  MicroLifeDeviceSDK
//
//  藍牙管理協議定義 (Bluetooth Manager Protocol Definition)
//  提供標準化的藍牙操作接口，實現解耦與模塊化
//

#import <Foundation/Foundation.h>
#import <CoreBluetooth/CoreBluetooth.h>
#import <MicroLifeDeviceSDK/BluetoothCallbacks.h>

NS_ASSUME_NONNULL_BEGIN

/**
 * 藍牙管理協議
 * Bluetooth Manager Protocol
 * 定義藍牙操作的標準接口，包括設備掃描、連接、服務發現等功能
 */
__attribute__((swift_attr("@MainActor")))
@protocol BluetoothManagerProtocol <NSObject>

#pragma mark - 掃描相關方法 (Scanning Methods)
/**
 * 開始掃描藍牙設備
 * Start scanning for bluetooth devices
 */
- (void)startScan;

/**
 * 停止掃描藍牙設備
 * Stop scanning for bluetooth devices
 */
- (void)cancelScan;

#pragma mark - 連接相關方法 (Connection Methods)
/**
 * 連接指定的藍牙設備
 * Connect to specified bluetooth device
 * @param device 要連接的設備 (device to connect)
 */
- (void)connectDevice:(id)device;

/**
 * 斷開指定設備的連接
 * Disconnect specified device
 * @param device 要斷開的設備 (device to disconnect)
 */
- (void)disconnectDevice:(id)device;

/**
 * 斷開所有已連接的設備
 * Disconnect all connected devices
 */
- (void)cancelAllConnect;

#pragma mark - 數據通信方法 (Data Communication Methods)
/**
 * 設置特徵值通知
 * Set characteristic notification
 * @param device 目標設備 (target device)
 * @param characteristic 目標特徵值 (target characteristic)
 */
- (void)setNotifiy:(id)device characteristic:(CBCharacteristic *)characteristic NS_SWIFT_NAME(setNotify(device:characteristic:));

/**
 * 寫入數據到設備
 * Write data to device
 * @param value 要寫入的數據 (data to write)
 * @param deviceInfo 目標設備信息 (target device info)
 * @return 是否寫入成功 (whether write successful)
 */
- (BOOL)writeValue:(NSString *)value DeviceInfo:(id)deviceInfo NS_SWIFT_NAME(writeValue(_:deviceInfo:));

/**
 * 心跳檢測
 * Heart beat check
 * @param value 心跳數據 (heart beat data)
 * @param deviceInfo 目標設備信息 (target device info)
 */
- (void)HEARTCHECK:(NSString *)value DeviceInfo:(id)deviceInfo NS_SWIFT_NAME(heartCheck(_:deviceInfo:));


#pragma mark - 回調設置方法 (Callback Setting Methods)
/**
 * 設置藍牙狀態更新回調
 * Set bluetooth state update callback
 */
- (void)getDidUpdateStateBlock:(didUpdateStateBlock)block NS_SWIFT_NAME(setDidUpdateStateBlock(_:));

/**
 * 設置掃描相關回調
 * Set scanning related callbacks
 */
- (void)getScanDeviceBlock:(scanDeviceBlock)scanDeviceBlock
           CancelScanBlock:(cancelScanBlock)cancelScanBlock NS_SWIFT_NAME(setScanCallbacks(scanDeviceBlock:cancelScanBlock:));

/**
 * 設置連接相關回調
 * Set connection related callbacks
 */
- (void)getConnectDeviceStateBlock:(connectDeviceStateBlock)connectDeviceStateBlock
   CancelAllDevicesConnectionBlock:(cancelAllDevicesConnectionBlock)cancelAllDevicesConnectionBlock NS_SWIFT_NAME(setConnectionCallbacks(connectDeviceStateBlock:cancelAllDevicesConnectionBlock:));

/**
 * 設置服務發現回調
 * Set service discovery callback
 */
- (void)getDiscoverServicesBlock:(discoverServicesBlock)block NS_SWIFT_NAME(setDiscoverServicesBlock(_:));

/**
 * 設置特徵值發現回調
 * Set characteristic discovery callback
 */
- (void)getDiscoverCharacteristicsBlock:(discoverCharacteristicsBlock)block NS_SWIFT_NAME(setDiscoverCharacteristicsBlock(_:));

/**
 * 設置描述符發現回調
 * Set descriptor discovery callback
 */
- (void)getDiscoverDescriptorsForCharacteristicBlock:(discoverDescriptorsForCharacteristicBlock)block NS_SWIFT_NAME(setDiscoverDescriptorsForCharacteristicBlock(_:));

/**
 * 設置特徵值通知狀態更新回調
 * Set characteristic notification state update callback
 */
- (void)getUpdateNotificationStateForCharacteristicBlock:(updateNotificationStateForCharacteristicBlock)block NS_SWIFT_NAME(setUpdateNotificationStateForCharacteristicBlock(_:));

/**
 * 設置讀取RSSI回調
 * Set read RSSI callback
 */
- (void)getReadRSSIBlock:(readRSSIBlock)block NS_SWIFT_NAME(setReadRSSIBlock(_:));

/**
 * 設置寫入特徵值回調
 * Set write characteristic callback
 */
- (void)getWriteValueForCharacteristicBlock:(writeValueBlock)block NS_SWIFT_NAME(setWriteValueForCharacteristicBlock(_:));

/**
 * 設置讀取特徵值回調
 * Set read characteristic callback
 */
- (void)getReadValueForCharacteristicBlock:(readValueBlock)block NS_SWIFT_NAME(setReadValueForCharacteristicBlock(_:));

/**
 * 設置讀取描述符回調
 * Set read descriptor callback
 */
- (void)getReadValueForDescriptorsBlock:(readValueForDescriptorsBlock)block NS_SWIFT_NAME(setReadValueForDescriptorsBlock(_:));

/**
 * 設置特徵值通知數據回調
 * Set characteristic notification data callback
 */
- (void)getReadNotifyValueForCharacteristicBlock:(readNotifyValueForCharacteristicBlock)block NS_SWIFT_NAME(setReadNotifyValueForCharacteristicBlock(_:));

@end

NS_ASSUME_NONNULL_END